{%- if cookiecutter.use_postgresql %}
"""Async PostgreSQL database session."""

from collections.abc import AsyncGenerator
from contextlib import asynccontextmanager

from sqlalchemy.ext.asyncio import AsyncSession, async_sessionmaker, create_async_engine

from app.core.config import settings

engine = create_async_engine(
    settings.DATABASE_URL,
    echo=settings.DEBUG,
    pool_size=settings.DB_POOL_SIZE,
    max_overflow=settings.DB_MAX_OVERFLOW,
    pool_timeout=settings.DB_POOL_TIMEOUT,
)

async_session_maker = async_sessionmaker(
    engine,
    class_=AsyncSession,
    expire_on_commit=False,
)


@asynccontextmanager
async def get_db_session() -> AsyncGenerator[AsyncSession, None]:
    """Get async database session.

    Can be used both as a FastAPI dependency and as an async context manager.
    """
    async with async_session_maker() as session:
        try:
            yield session
            await session.commit()
        except Exception:
            await session.rollback()
            raise
        finally:
            await session.close()


async def close_db() -> None:
    """Close database connections."""
    await engine.dispose()


{%- elif cookiecutter.use_mongodb %}
"""Async MongoDB database session."""

from motor.motor_asyncio import AsyncIOMotorClient, AsyncIOMotorDatabase

from app.core.config import settings

client: AsyncIOMotorClient | None = None


async def get_db_session() -> AsyncIOMotorDatabase:
    """Get MongoDB database instance."""
    global client
    if client is None:
        client = AsyncIOMotorClient(settings.MONGO_URL)
    return client[settings.MONGO_DB]


async def close_db() -> None:
    """Close MongoDB connection."""
    global client
    if client is not None:
        client.close()
        client = None


{%- elif cookiecutter.use_sqlite %}
"""Sync SQLite database session."""

from collections.abc import Generator

from sqlalchemy import create_engine
from sqlalchemy.orm import Session, sessionmaker

from app.core.config import settings

engine = create_engine(
    settings.DATABASE_URL,
    connect_args={"check_same_thread": False},
    echo=settings.DEBUG,
)

SessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)


def get_db_session() -> Generator[Session, None, None]:
    """Get sync database session."""
    db = SessionLocal()
    try:
        yield db
        db.commit()
    except Exception:
        db.rollback()
        raise
    finally:
        db.close()


{%- else %}
"""No database configured."""


async def get_db_session():
    """No-op when database is disabled."""
    yield None
{%- endif %}
